<?PHP  if ( ! defined('BASEPATH')) exit('No direct script access allowed'); 

class ImapSocket {
	private $_socket;
	public function __construct($options) {
		$tls = isset($options['tls']) ? $options['tls'] : TRUE; //always use TLS if not specified
		$this->_socket = $this->_connect(IMAP_HOSTNAME, IMAP_PORT, $tls); 
		$this->_login($options['login'], $options['password']);
		if(!is_null($options['mailbox'])) { $this->select_mailbox($options['mailbox']); }
	}
  
	public function __destruct() {
		fclose($this->_socket);
	}
  
	/* This function fetches the response from the socket, removing the * label from the response
	 */
	private function _gets() {
		$result = array();
		while(substr($str = fgets($this->_socket), 0, 1) == '*') { 
			$result[] = substr($str, 0, -2);
		}
		$result[] = substr($str, 0, -2);
		return $result;
	}
	
	/* This function sends the $cmd parameter as a command to the IMAP server, 
	 * the $uid parameter serves as the unique label for the command
	 */
	private function _send ($cmd, $uid = '.') {
		$query = "$uid $cmd\r\n";
		$count = fwrite($this->_socket, $query);
		if($count === strlen($query)) {
			return $this->_gets();
		}
		else { throw new Exception('An error occurred while executing command: '.$cmd); }
	}

	/* This function establishes a raw socket connection to the IMAP server
	 * specified by the $server parameter, over the port specified in the $port parameter.
	 * Optional parameter for TLS takes a boolean value to decide whether to attempt the connection
	 * over TLS (TLS must be supported by the IMAP server).
	 */
	private function _connect($server, $port, $tls) {
		$context = stream_context_create(array(
			'ssl' => array('SNI_server_name' => $server),
		));
		if($tls) {
			//try to open stream using TLS, ignore warnings if it fails since we'll try the STARTTLS next
			$fd = @stream_socket_client('tls://'.$server.':'.$port, $errno, $errstr, 2, STREAM_CLIENT_CONNECT, $context);
			//if TLS isn't set to start immediately, attempt STARTTLS over TCP
			if(!$fd) {
				$fd = stream_socket_client('tcp://'.$server.':'.$port, $errno, $errstr, 2, STREAM_CLIENT_CONNECT, $context);
				stream_set_timeout($fd,1);
				$result = fgets($fd);
				if(strpos($result,'STARTTLS') !== FALSE) {
					$count = fwrite($fd,". STARTTLS\r\n");
					fgets($fd); //must get response to clear it before doing STARTTLS
					stream_socket_enable_crypto($fd, true, STREAM_CRYPTO_METHOD_TLS_CLIENT);
				}
				else { $errno = TRUE; }
			}
		}
		else {
			$fd = stream_socket_client('tcp://'.$server.':'.$port, $errno, $errstr, 2, STREAM_CLIENT_CONNECT, $context);
		}
		if(!$errno) { return $fd; }
		else { throw new Exception('A connection to the IMAP server could not be established.'); }
	}
	
	/* This function sends a login command to the IMAP server over the socket connection
	 * utilizing the send function, which sends a command over the socket connection.
	 */
	private function _login($login, $password) {
		$result = $this->_send("LOGIN $login \"$password\"");
		$result = array_pop($result);
		if(strpos($result,'Logged in.') !== FALSE) { throw new Exception('Login to IMAP Failed'); }
	}
	
	/* This function sends a select mailbox command to the IMAP server over the socket connection
	 * utilizing the send function, which sends a command over the socket connection.
	 */
	public function select_mailbox($mailbox) {
		$result = $this->_send("SELECT \"$mailbox\"");
		$result = array_pop($result);
		if($result != ". OK [READ-WRITE] Select completed.") {
			throw new Exception('Could not select IMAP mailbox.');
		}
	}
	
	/* This function retrieves the flags on a message (specified by the $uid parameter, which is the message unique id)
     * from the IMAP server over the socket connection utilizing the send function, 
	 * which sends a command over the socket connection. The returned result
	 * is parsed using a regular expression to get the flags into an array, which is returned. If no flags are
	 * found, an empty array is returned.
	 */
	public function get_flags($uid) {
		$result = $this->_send("UID FETCH $uid (FLAGS)");
		preg_match_all("#\\* \\d+ FETCH \\(UID [0-9]* FLAGS \\((.*)\\)\\)#", $result[0], $matches);
		if(isset ($matches[1][0])) { 
			return explode(' ', $matches[1][0]);
		}
		else { return array(); }
	}
}